<?php
/**
 * OptiCore SaaS - CajaController
 */

class CajaController
{
    private Caja $model;

    public function __construct()
    {
        Auth::requirePermission('caja.ver');
        $this->model = new Caja();
    }

    // ── GET /caja ─────────────────────────────────────────────
    public function index(): void
    {
        $sucursalId  = Auth::sucursalId();
        $page        = currentPage();
        $cajaAbierta = $sucursalId ? $this->model->getSesionAbierta($sucursalId) : null;
        $movimientos = [];
        $resumen     = [];

        if ($cajaAbierta) {
            $movimientos = $this->model->getMovimientosSesion((int) $cajaAbierta['id']);
            $resumen     = $this->model->getResumenSesion((int) $cajaAbierta['id']);
        }

        $sesiones = $this->model->listarSesiones($sucursalId, $page, ITEMS_PER_PAGE);

        view('caja.index', [
            'title'       => 'Caja',
            'cajaAbierta' => $cajaAbierta,
            'movimientos' => $movimientos,
            'resumen'     => $resumen,
            'historial'   => $sesiones['data'],
            'pagination'  => $sesiones['pagination'],
        ]);
    }

    // ── GET /caja/apertura ────────────────────────────────────
    public function apertura(): void
    {
        Auth::requirePermission('caja.abrir');

        $sucursalId = Auth::sucursalId();
        if (!$sucursalId) {
            flash('error', 'No tienes sucursal asignada.');
            redirect('/caja');
        }

        // Verificar si ya hay caja abierta
        $cajaAbierta = $this->model->getSesionAbierta($sucursalId);
        if ($cajaAbierta) {
            flash('info', 'Ya tienes una caja abierta.');
            redirect('/caja');
        }

        $sucursales = (new Sucursal())->getByEmpresa(Auth::empresaId());
        $ultimaCaja = $this->model->getUltimaCajaCerrada($sucursalId);

        view('caja.apertura', [
            'title'      => 'Apertura de Caja',
            'sucursales' => $sucursales,
            'ultimaCaja' => $ultimaCaja,
        ]);
    }

    // ── POST /caja/apertura ───────────────────────────────────
    public function abrir(): void
    {
        Auth::requirePermission('caja.abrir');
        csrf_verify();

        $sucursalId = Auth::sucursalId();
        $empresaId  = Auth::empresaId();

        if (!$sucursalId || !$empresaId) {
            flash('error', 'No tienes sucursal asignada.');
            redirect('/caja');
        }

        // Verificar si ya hay caja abierta
        if ($this->model->getSesionAbierta($sucursalId)) {
            flash('error', 'Ya existe una caja abierta para esta sucursal.');
            redirect('/caja');
        }

        $montoApertura = (float) ($_POST['saldo_inicial'] ?? $_POST['monto_apertura'] ?? 0);
        $observaciones = $_POST['observaciones'] ?? '';

        try {
            $sesionId = $this->model->abrir($empresaId, $sucursalId, Auth::id(), $montoApertura, $observaciones);
            AuditLog::log('apertura_caja', 'caja_sesiones', $sesionId);
            flash('success', 'Caja abierta correctamente.');
            redirect('/caja');
        } catch (Exception $e) {
            flash('error', 'Error al abrir caja: ' . $e->getMessage());
            redirect('/caja/apertura');
        }
    }

    // ── GET /caja/cierre ──────────────────────────────────────
    public function cierre(): void
    {
        Auth::requirePermission('caja.cerrar');

        $sucursalId = Auth::sucursalId();
        if (!$sucursalId) {
            flash('error', 'No tienes sucursal asignada.');
            redirect('/caja');
        }

        $caja = $this->model->getSesionAbierta($sucursalId);
        if (!$caja) {
            flash('error', 'No hay caja abierta para cerrar.');
            redirect('/caja');
        }

        $resumen       = $this->model->getResumenSesion((int) $caja['id']);
        $porMetodo     = $this->model->getResumenPorFormaPago((int) $caja['id']);
        $saldoEsperado = $caja['monto_apertura'] + ($resumen['total_ingresos'] ?? 0) - ($resumen['total_egresos'] ?? 0);

        view('caja.cierre', [
            'title'         => 'Cierre de Caja',
            'caja'          => $caja,
            'resumen'       => $resumen,
            'porMetodo'     => $porMetodo,
            'saldoEsperado' => $saldoEsperado,
        ]);
    }

    // ── POST /caja/cierre ─────────────────────────────────────
    public function cerrar(): void
    {
        Auth::requirePermission('caja.cerrar');
        csrf_verify();

        $sucursalId = Auth::sucursalId();
        $sesion     = $this->model->getSesionAbierta($sucursalId);

        if (!$sesion) {
            flash('error', 'No hay caja abierta para cerrar.');
            redirect('/caja');
        }

        $montoCierreReal = (float) ($_POST['saldo_final'] ?? $_POST['monto_cierre_real'] ?? 0);
        $observaciones   = $_POST['observaciones'] ?? null;

        try {
            $this->model->cerrar((int) $sesion['id'], $montoCierreReal, $observaciones);
            AuditLog::log('cierre_caja', 'caja_sesiones', (int) $sesion['id']);
            flash('success', 'Caja cerrada correctamente.');
            redirect('/caja/' . $sesion['id']);
        } catch (Exception $e) {
            flash('error', 'Error al cerrar caja: ' . $e->getMessage());
            redirect('/caja/cierre');
        }
    }

    // ── GET /caja/{id} ────────────────────────────────────────
    public function show(int $id): void
    {
        $caja = $this->model->getSesionById($id);
        if (!$caja) {
            flash('error', 'Sesión de caja no encontrada.');
            redirect('/caja');
        }

        $movimientos = $this->model->getMovimientosSesion($id);
        $resumen     = $this->model->getResumenSesion($id);
        $porMetodo   = $this->model->getResumenPorFormaPago($id);
        $ventas      = $this->model->getVentasSesion($id);

        view('caja.show', [
            'title'       => 'Caja #' . $caja['correlativo'],
            'caja'        => $caja,
            'movimientos' => $movimientos,
            'resumen'     => $resumen,
            'porMetodo'   => $porMetodo,
            'ventas'      => $ventas,
        ]);
    }

    // ── POST /caja/movimiento ─────────────────────────────────
    public function movimiento(): void
    {
        Auth::requirePermission('caja.movimiento');
        csrf_verify();

        $sucursalId = Auth::sucursalId();
        $sesion     = $this->model->getSesionAbierta($sucursalId);

        if (!$sesion) {
            if (isAjax()) {
                jsonResponse(['error' => 'No hay caja abierta.'], 400);
            }
            flash('error', 'No hay caja abierta.');
            redirect('/caja');
        }

        $errors = validateRequired($_POST, [
            'tipo'    => 'Tipo',
            'concepto'=> 'Concepto',
            'monto'   => 'Monto',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/caja/' . $sesion['id']);
        }

        $monto = (float) $_POST['monto'];
        if ($monto <= 0) {
            flash('error', 'El monto debe ser mayor a 0.');
            redirect('/caja/' . $sesion['id']);
        }

        try {
            $this->model->registrarMovimiento([
                'empresa_id'  => Auth::empresaId(),
                'sucursal_id' => $sucursalId,
                'sesion_id'   => (int) $sesion['id'],
                'usuario_id'  => Auth::id(),
                'tipo'        => $_POST['tipo'],
                'concepto'    => $_POST['concepto'],
                'monto'       => $monto,
            ]);
            flash('success', 'Movimiento registrado correctamente.');
        } catch (Exception $e) {
            flash('error', 'Error al registrar movimiento: ' . $e->getMessage());
        }

        redirect('/caja/' . $sesion['id']);
    }
}
