<?php
/**
 * OptiCore SaaS - DashboardController
 */

class DashboardController
{
    // ── GET /dashboard ────────────────────────────────────────
    public function index(): void
    {
        Auth::requireAuth();

        $data = [];

        if (Auth::isSuperAdmin()) {
            // ── Estadísticas SuperAdmin ───────────────────────
            $db = db();
            $data['stats'] = [
                'empresas'   => $db->count('empresas'),
                'planes'     => $db->count('planes'),
                'licencias'  => $db->count('licencias', ['estado' => 'activa']),
                'usuarios'   => $db->count('usuarios'),
            ];
            $data['empresas_recientes'] = $db->fetchAll(
                "SELECT e.*, p.nombre as plan_nombre
                 FROM empresas e
                 JOIN planes p ON e.plan_id = p.id
                 ORDER BY e.created_at DESC LIMIT 5"
            );
            $data['licencias_por_vencer'] = (new Licencia())->getProximasVencer(7);

        } else {
            // ── Estadísticas por empresa ──────────────────────
            $empresaId  = Auth::empresaId();
            $sucursalId = Auth::sucursalId();
            $db         = db();

            $data['stats'] = [
                'pacientes'  => $db->count('pacientes',  ['empresa_id' => $empresaId]),
                'productos'  => $db->count('productos',  ['empresa_id' => $empresaId]),
                'ventas_hoy' => $this->ventasHoy($empresaId, $sucursalId),
                'caja'       => $this->cajaAbierta($empresaId, $sucursalId),
            ];

            // Ventas últimos 7 días
            $data['ventas_semana'] = $db->fetchAll(
                "SELECT DATE(fecha) as dia, COUNT(*) as total_ventas, SUM(total) as monto
                 FROM ventas
                 WHERE empresa_id = ? AND estado = 'confirmada'
                   AND fecha >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                 GROUP BY DATE(fecha)
                 ORDER BY dia ASC",
                [$empresaId]
            );

            // Últimas ventas
            $data['ultimas_ventas'] = $db->fetchAll(
                "SELECT v.*, CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                        CONCAT(u.nombre,' ',u.apellido) as usuario_nombre
                 FROM ventas v
                 LEFT JOIN pacientes p ON v.paciente_id = p.id
                 JOIN usuarios u ON v.usuario_id = u.id
                 WHERE v.empresa_id = ?
                 ORDER BY v.created_at DESC LIMIT 5",
                [$empresaId]
            );

            // Productos con stock bajo
            $data['stock_bajo'] = $db->fetchAll(
                "SELECT pr.nombre, pr.stock_minimo, s.cantidad, su.nombre as sucursal
                 FROM stock s
                 JOIN productos pr ON s.producto_id = pr.id
                 JOIN sucursales su ON s.sucursal_id = su.id
                 WHERE s.empresa_id = ? AND s.cantidad <= pr.stock_minimo AND pr.stock_minimo > 0
                 ORDER BY s.cantidad ASC LIMIT 5",
                [$empresaId]
            );
        }

        $data['title'] = 'Dashboard';
        view('dashboard.index', $data);
    }

    // ── Ventas del día ────────────────────────────────────────
    private function ventasHoy(int $empresaId, ?int $sucursalId): array
    {
        $db  = db();
        $sql = "SELECT COUNT(*) as total, COALESCE(SUM(total),0) as monto
                FROM ventas
                WHERE empresa_id = ? AND estado = 'confirmada' AND DATE(fecha) = CURDATE()";
        $params = [$empresaId];

        if ($sucursalId) {
            $sql    .= " AND sucursal_id = ?";
            $params[] = $sucursalId;
        }

        return $db->fetchOne($sql, $params) ?: ['total' => 0, 'monto' => 0];
    }

    // ── Estado de caja ────────────────────────────────────────
    private function cajaAbierta(int $empresaId, ?int $sucursalId): array|false
    {
        $db  = db();
        $sql = "SELECT * FROM caja_sesiones
                WHERE empresa_id = ? AND estado = 'abierta'";
        $params = [$empresaId];

        if ($sucursalId) {
            $sql    .= " AND sucursal_id = ?";
            $params[] = $sucursalId;
        }

        $sql .= " ORDER BY fecha_apertura DESC LIMIT 1";
        return $db->fetchOne($sql, $params);
    }
}
