<?php
/**
 * OptiCore SaaS - LaboratorioController
 * Gestión de Órdenes de Laboratorio
 */

class LaboratorioController
{
    private OrdenLaboratorio $model;

    public function __construct()
    {
        Auth::requirePermission('laboratorio.ver');
        $this->model = new OrdenLaboratorio();
    }

    // ── GET /laboratorio ──────────────────────────────────────
    public function index(): void
    {
        $page     = currentPage();
        $filtros  = [
            'estado' => get('estado'),
            'desde'  => get('desde'),
            'hasta'  => get('hasta'),
            'q'      => get('q'),
        ];
        $resultado = $this->model->getAll($page, array_filter($filtros));

        view('laboratorio.index', [
            'title'    => 'Órdenes de Laboratorio',
            'ordenes'  => $resultado['data'],
            'pagination' => $resultado,
            'filtros'  => $filtros,
        ]);
    }

    // ── GET /laboratorio/create ───────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('laboratorio.crear');

        $proveedores = (new Proveedor())->getActivos('laboratorio');

        view('laboratorio.create', [
            'title'       => 'Nueva Orden de Laboratorio',
            'proveedores' => $proveedores,
        ]);
    }

    // ── POST /laboratorio ─────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('laboratorio.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'fecha_recepcion' => 'Fecha de recepción',
            'descripcion'     => 'Descripción del trabajo',
        ]);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/laboratorio/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_orden_laboratorio', 'ordenes_laboratorio', $id);
        flash('success', 'Orden de laboratorio creada correctamente.');
        redirect("/laboratorio/$id");
    }

    // ── GET /laboratorio/{id} ─────────────────────────────────
    public function show(int $id): void
    {
        $orden = $this->model->getById($id);
        if (!$orden) {
            flash('error', 'Orden no encontrada.');
            redirect('/laboratorio');
        }

        view('laboratorio.show', [
            'title' => 'Orden ' . $orden['correlativo'],
            'orden' => $orden,
        ]);
    }

    // ── GET /laboratorio/{id}/edit ────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('laboratorio.editar');

        $orden = $this->model->getById($id);
        if (!$orden) {
            flash('error', 'Orden no encontrada.');
            redirect('/laboratorio');
        }

        $proveedores = (new Proveedor())->getActivos('laboratorio');

        view('laboratorio.edit', [
            'title'       => 'Editar Orden ' . $orden['correlativo'],
            'orden'       => $orden,
            'proveedores' => $proveedores,
        ]);
    }

    // ── POST /laboratorio/{id} ────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('laboratorio.editar');
        csrf_verify();

        $orden = $this->model->getById($id);
        if (!$orden) {
            flash('error', 'Orden no encontrada.');
            redirect('/laboratorio');
        }

        $errors = validateRequired($_POST, ['descripcion' => 'Descripción del trabajo']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/laboratorio/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_orden_laboratorio', 'ordenes_laboratorio', $id, $orden, $_POST);
        flash('success', 'Orden actualizada correctamente.');
        redirect("/laboratorio/$id");
    }

    // ── POST /laboratorio/{id}/estado ─────────────────────────
    public function cambiarEstado(int $id): void
    {
        Auth::requirePermission('laboratorio.editar');
        csrf_verify();

        $orden = $this->model->getById($id);
        if (!$orden) {
            flash('error', 'Orden no encontrada.');
            redirect('/laboratorio');
        }

        $estado = $_POST['estado'] ?? '';
        $estadosValidos = ['recibido', 'en_proceso', 'listo', 'entregado', 'cancelado'];
        if (!in_array($estado, $estadosValidos)) {
            flash('error', 'Estado no válido.');
            redirect("/laboratorio/$id");
        }

        $this->model->cambiarEstado($id, $estado);
        AuditLog::log('cambiar_estado_laboratorio', 'ordenes_laboratorio', $id, $orden, ['estado' => $estado]);
        flash('success', 'Estado actualizado correctamente.');
        redirect("/laboratorio/$id");
    }
}
