<?php
/**
 * OptiCore SaaS - Caja Model
 */

class Caja extends BaseModel
{
    protected string $table = 'caja_sesiones';

    public function getSesionAbierta(int $sucursalId): array|false
    {
        return $this->db->fetchOne(
            "SELECT cs.*, CONCAT(u.nombre,' ',u.apellido) as cajero_nombre,
                    s.nombre as sucursal_nombre
             FROM caja_sesiones cs
             LEFT JOIN usuarios u ON cs.usuario_id = u.id
             LEFT JOIN sucursales s ON cs.sucursal_id = s.id
             WHERE cs.sucursal_id = ? AND cs.empresa_id = ? AND cs.estado = 'abierta'
             ORDER BY cs.fecha_apertura DESC LIMIT 1",
            [$sucursalId, $this->empresaScope()]
        );
    }

    // ── Listar sesiones con estructura ['data','pagination'] ───
    public function listarSesiones(?int $sucursalId = null, int $page = 1, int $perPage = ITEMS_PER_PAGE): array
    {
        $result = $this->getHistorial($page, $sucursalId);
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    // ── Alias para getById ─────────────────────────────────────
    public function getSesionById(int $id): array|false
    {
        return $this->getById($id);
    }

    // ── Última caja cerrada de una sucursal ────────────────────
    public function getUltimaCajaCerrada(int $sucursalId): array|false
    {
        return $this->db->fetchOne(
            "SELECT cs.*, CONCAT(u.nombre,' ',u.apellido) as cajero_nombre
             FROM caja_sesiones cs
             LEFT JOIN usuarios u ON cs.usuario_id = u.id
             WHERE cs.sucursal_id = ? AND cs.empresa_id = ? AND cs.estado = 'cerrada'
             ORDER BY cs.fecha_cierre DESC LIMIT 1",
            [$sucursalId, $this->empresaScope()]
        );
    }

    // ── Alias para getMovimientos ──────────────────────────────
    public function getMovimientosSesion(int $sesionId): array
    {
        return $this->getMovimientos($sesionId);
    }

    // ── Ventas registradas durante una sesión ──────────────────
    public function getVentasSesion(int $sesionId): array
    {
        $sesion = $this->getById($sesionId);
        if (!$sesion) return [];

        $params = [$sesion['sucursal_id'], $sesion['empresa_id'], $sesion['fecha_apertura']];
        $hasta  = $sesion['fecha_cierre'] ?? date('Y-m-d H:i:s');
        $params[] = $hasta;

        return $this->db->fetchAll(
            "SELECT v.*, CONCAT(COALESCE(p.nombre,''), ' ', COALESCE(p.apellido,'')) as paciente_nombre
             FROM ventas v
             LEFT JOIN pacientes p ON v.paciente_id = p.id
             WHERE v.sucursal_id = ? AND v.empresa_id = ?
               AND v.fecha >= ? AND v.fecha <= ?
             ORDER BY v.fecha DESC",
            $params
        );
    }

    // ── Resumen completo de una sesión ─────────────────────────
    public function getResumenSesion(int $sesionId): array
    {
        $sesion   = $this->getById($sesionId);
        $apertura = $sesion ? (float) $sesion['monto_apertura'] : 0;

        $ingresos = (float) $this->db->fetchColumn(
            "SELECT COALESCE(SUM(monto), 0) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'ingreso'",
            [$sesionId]
        );
        $egresos = (float) $this->db->fetchColumn(
            "SELECT COALESCE(SUM(monto), 0) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'egreso'",
            [$sesionId]
        );
        $totalVentas = (float) $this->db->fetchColumn(
            "SELECT COALESCE(SUM(monto), 0) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'ingreso' AND referencia_tipo = 'venta'",
            [$sesionId]
        );
        $cantidadVentas = (int) $this->db->fetchColumn(
            "SELECT COUNT(*) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'ingreso' AND referencia_tipo = 'venta'",
            [$sesionId]
        );

        return [
            'monto_apertura'  => $apertura,
            'total_ingresos'  => $ingresos,
            'total_egresos'   => $egresos,
            'total_sistema'   => $apertura + $ingresos - $egresos,
            'total_ventas'    => $totalVentas,
            'cantidad_ventas' => $cantidadVentas,
            'por_forma_pago'  => $this->getResumenPorFormaPago($sesionId),
        ];
    }

    // ── Registrar movimiento manual ────────────────────────────
    public function registrarMovimiento(array $data): int
    {
        return $this->db->insert('caja_movimientos', array_merge([
            'fecha' => date('Y-m-d H:i:s'),
        ], $data));
    }

    // ── Abrir caja (firma compatible con CajaController) ──────
    public function abrir(int $empresaId, int $sucursalId, int $userId, float $montoApertura, string $obs = ''): int
    {
        $db = $this->db;

        // Obtener correlativo
        $sucursal    = $db->fetchOne("SELECT correlativo_caja FROM sucursales WHERE id = ?", [$sucursalId]);
        $correlativo = generarCorrelativo('C-' . str_pad($sucursalId, 3, '0', STR_PAD_LEFT) . '-', $sucursal['correlativo_caja'] ?? 0);
        $db->query("UPDATE sucursales SET correlativo_caja = correlativo_caja + 1 WHERE id = ?", [$sucursalId]);

        $id = $db->insert('caja_sesiones', [
            'empresa_id'     => $empresaId,
            'sucursal_id'    => $sucursalId,
            'usuario_id'     => $userId,
            'correlativo'    => $correlativo,
            'fecha_apertura' => date('Y-m-d H:i:s'),
            'monto_apertura' => $montoApertura,
            'estado'         => 'abierta',
            'observaciones'  => $obs,
        ]);

        AuditLog::log('apertura_caja', 'caja_sesiones', $id, null, ['monto' => $montoApertura]);
        return $id;
    }

    public function cerrar(int $sesionId, float $montoCierreReal, string $obs = ''): bool
    {
        $sesion = $this->getById($sesionId);
        if (!$sesion || $sesion['estado'] !== 'abierta') return false;

        // Calcular total sistema
        $totalSistema = $this->calcularTotalSistema($sesionId, $sesion['monto_apertura']);
        $diferencia   = $montoCierreReal - $totalSistema;

        $this->db->update('caja_sesiones', [
            'fecha_cierre'         => date('Y-m-d H:i:s'),
            'monto_cierre_real'    => $montoCierreReal,
            'monto_cierre_sistema' => $totalSistema,
            'diferencia'           => $diferencia,
            'estado'               => 'cerrada',
            'observaciones'        => $obs,
        ], ['id' => $sesionId]);

        AuditLog::log('cierre_caja', 'caja_sesiones', $sesionId, $sesion, [
            'monto_real'    => $montoCierreReal,
            'monto_sistema' => $totalSistema,
            'diferencia'    => $diferencia,
        ]);
        return true;
    }

    public function calcularTotalSistema(int $sesionId, float $montoApertura): float
    {
        $ingresos = (float) $this->db->fetchColumn(
            "SELECT COALESCE(SUM(monto), 0) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'ingreso'",
            [$sesionId]
        );
        $egresos = (float) $this->db->fetchColumn(
            "SELECT COALESCE(SUM(monto), 0) FROM caja_movimientos WHERE sesion_id = ? AND tipo = 'egreso'",
            [$sesionId]
        );
        return $montoApertura + $ingresos - $egresos;
    }

    public function getMovimientos(int $sesionId): array
    {
        return $this->db->fetchAll(
            "SELECT cm.*, fp.nombre as forma_pago_nombre,
                    CONCAT(u.nombre,' ',u.apellido) as usuario_nombre
             FROM caja_movimientos cm
             LEFT JOIN formas_pago fp ON cm.forma_pago_id = fp.id
             LEFT JOIN usuarios u ON cm.usuario_id = u.id
             WHERE cm.sesion_id = ?
             ORDER BY cm.fecha ASC",
            [$sesionId]
        );
    }

    public function getResumenPorFormaPago(int $sesionId): array
    {
        return $this->db->fetchAll(
            "SELECT fp.nombre as forma_pago, fp.tipo,
                    SUM(CASE WHEN cm.tipo='ingreso' THEN cm.monto ELSE 0 END) as ingresos,
                    SUM(CASE WHEN cm.tipo='egreso' THEN cm.monto ELSE 0 END) as egresos
             FROM caja_movimientos cm
             LEFT JOIN formas_pago fp ON cm.forma_pago_id = fp.id
             WHERE cm.sesion_id = ?
             GROUP BY cm.forma_pago_id
             ORDER BY fp.nombre",
            [$sesionId]
        );
    }

    public function agregarMovimiento(int $sesionId, string $tipo, string $concepto, float $monto, ?int $formaPagoId = null): int
    {
        $sesion = $this->getById($sesionId);
        return $this->db->insert('caja_movimientos', [
            'empresa_id'    => $this->empresaScope(),
            'sucursal_id'   => $sesion['sucursal_id'],
            'sesion_id'     => $sesionId,
            'usuario_id'    => Auth::id(),
            'tipo'          => $tipo,
            'concepto'      => $concepto,
            'monto'         => $monto,
            'forma_pago_id' => $formaPagoId,
            'fecha'         => date('Y-m-d H:i:s'),
        ]);
    }

    public function getHistorial(int $page = 1, ?int $sucursalId = null): array
    {
        $params = [$this->empresaScope()];
        $where  = 'cs.empresa_id = ?';
        if ($sucursalId) {
            $where   .= ' AND cs.sucursal_id = ?';
            $params[] = $sucursalId;
        }

        $sql = "SELECT cs.*, CONCAT(u.nombre,' ',u.apellido) as cajero_nombre,
                       s.nombre as sucursal_nombre
                FROM caja_sesiones cs
                LEFT JOIN usuarios u ON cs.usuario_id = u.id
                LEFT JOIN sucursales s ON cs.sucursal_id = s.id
                WHERE $where ORDER BY cs.fecha_apertura DESC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT cs.*, CONCAT(u.nombre,' ',u.apellido) as cajero_nombre,
                    s.nombre as sucursal_nombre, e.nombre as empresa_nombre
             FROM caja_sesiones cs
             LEFT JOIN usuarios u ON cs.usuario_id = u.id
             LEFT JOIN sucursales s ON cs.sucursal_id = s.id
             LEFT JOIN empresas e ON cs.empresa_id = e.id
             WHERE cs.id = ? AND cs.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }
}
