<?php
/**
 * OptiCore SaaS - OrdenLaboratorio Model
 */

class OrdenLaboratorio extends BaseModel
{
    protected string $table = 'ordenes_laboratorio';

    public function getAll(int $page = 1, array $filtros = []): array
    {
        $params = [$this->empresaScope()];
        $where  = 'ol.empresa_id = ?';

        if (!empty($filtros['estado'])) {
            $where   .= ' AND ol.estado = ?';
            $params[] = $filtros['estado'];
        }
        if (!empty($filtros['desde'])) {
            $where   .= ' AND ol.fecha_recepcion >= ?';
            $params[] = $filtros['desde'];
        }
        if (!empty($filtros['hasta'])) {
            $where   .= ' AND ol.fecha_recepcion <= ?';
            $params[] = $filtros['hasta'];
        }
        if (!empty($filtros['q'])) {
            $where   .= ' AND (ol.correlativo LIKE ? OR CONCAT(p.nombre, \' \', p.apellido) LIKE ?)';
            $s        = '%' . $filtros['q'] . '%';
            $params[] = $s; $params[] = $s;
        }

        $sql = "SELECT ol.*,
                       CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre,
                       pr.nombre AS proveedor_nombre,
                       CONCAT(u.nombre, ' ', u.apellido) AS usuario_nombre,
                       s.nombre AS sucursal_nombre
                FROM `{$this->table}` ol
                LEFT JOIN pacientes p   ON ol.paciente_id  = p.id
                LEFT JOIN proveedores pr ON ol.proveedor_id = pr.id
                LEFT JOIN usuarios u    ON ol.usuario_id   = u.id
                LEFT JOIN sucursales s  ON ol.sucursal_id  = s.id
                WHERE $where
                ORDER BY ol.created_at DESC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT ol.*,
                    CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre,
                    p.telefono AS paciente_telefono,
                    pr.nombre AS proveedor_nombre,
                    pr.telefono AS proveedor_telefono,
                    CONCAT(u.nombre, ' ', u.apellido) AS usuario_nombre,
                    s.nombre AS sucursal_nombre,
                    v.correlativo AS venta_correlativo
             FROM `{$this->table}` ol
             LEFT JOIN pacientes p    ON ol.paciente_id  = p.id
             LEFT JOIN proveedores pr ON ol.proveedor_id = pr.id
             LEFT JOIN usuarios u     ON ol.usuario_id   = u.id
             LEFT JOIN sucursales s   ON ol.sucursal_id  = s.id
             LEFT JOIN ventas v       ON ol.venta_id     = v.id
             WHERE ol.id = ? AND ol.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function crear(array $data): int
    {
        $sucursalId = Auth::sucursalId() ?? $data['sucursal_id'] ?? null;
        $empresaId  = $this->empresaScope();

        // Generar correlativo
        $correlativo = $this->generarCorrelativo($empresaId);

        return $this->db->insert($this->table, [
            'empresa_id'      => $empresaId,
            'sucursal_id'     => $sucursalId,
            'venta_id'        => !empty($data['venta_id'])    ? (int) $data['venta_id']    : null,
            'receta_id'       => !empty($data['receta_id'])   ? (int) $data['receta_id']   : null,
            'paciente_id'     => !empty($data['paciente_id']) ? (int) $data['paciente_id'] : null,
            'proveedor_id'    => !empty($data['proveedor_id'])? (int) $data['proveedor_id']: null,
            'usuario_id'      => Auth::id(),
            'correlativo'     => $correlativo,
            'estado'          => 'recibido',
            'fecha_recepcion' => $data['fecha_recepcion'] ?? date('Y-m-d'),
            'fecha_estimada'  => !empty($data['fecha_estimada']) ? $data['fecha_estimada'] : null,
            'costo'           => $data['costo'] ?? 0,
            'descripcion'     => trim($data['descripcion'] ?? '') ?: null,
            'observaciones'   => trim($data['observaciones'] ?? '') ?: null,
        ]);
    }

    public function actualizar(int $id, array $data): int
    {
        return $this->update($id, [
            'proveedor_id'   => !empty($data['proveedor_id']) ? (int) $data['proveedor_id'] : null,
            'estado'         => $data['estado'] ?? 'recibido',
            'fecha_estimada' => !empty($data['fecha_estimada']) ? $data['fecha_estimada'] : null,
            'fecha_entrega'  => !empty($data['fecha_entrega'])  ? $data['fecha_entrega']  : null,
            'costo'          => $data['costo'] ?? 0,
            'descripcion'    => trim($data['descripcion'] ?? '') ?: null,
            'observaciones'  => trim($data['observaciones'] ?? '') ?: null,
        ]);
    }

    public function cambiarEstado(int $id, string $estado): int
    {
        $campos = ['estado' => $estado];
        if ($estado === 'entregado') {
            $campos['fecha_entrega'] = date('Y-m-d');
        }
        return $this->update($id, $campos);
    }

    public function getPendientes(): array
    {
        return $this->db->fetchAll(
            "SELECT ol.*, CONCAT(p.nombre, ' ', p.apellido) AS paciente_nombre
             FROM `{$this->table}` ol
             LEFT JOIN pacientes p ON ol.paciente_id = p.id
             WHERE ol.empresa_id = ? AND ol.estado NOT IN ('entregado','cancelado')
             ORDER BY ol.fecha_estimada ASC",
            [$this->empresaScope()]
        );
    }

    private function generarCorrelativo(int $empresaId): string
    {
        $ultimo = $this->db->fetchColumn(
            "SELECT COUNT(*) FROM `{$this->table}` WHERE empresa_id = ?",
            [$empresaId]
        );
        return 'LAB-' . str_pad((int)$ultimo + 1, 5, '0', STR_PAD_LEFT);
    }
}
