<?php
/**
 * OptiCore SaaS - Paciente Model
 */

class Paciente extends BaseModel
{
    protected string $table = 'pacientes';

    public function getAll(int $page = 1, string $search = '', ?int $sucursalId = null): array
    {
        $params = [$this->empresaScope()];
        $where  = 'p.empresa_id = ?';

        if ($search) {
            $where   .= " AND (CONCAT(p.nombre,' ',p.apellido) LIKE ? OR p.rut LIKE ? OR p.telefono LIKE ? OR p.email LIKE ?)";
            $s        = "%$search%";
            $params[] = $s; $params[] = $s; $params[] = $s; $params[] = $s;
        }
        if ($sucursalId) {
            $where   .= ' AND p.sucursal_id = ?';
            $params[] = $sucursalId;
        }

        $sql = "SELECT p.*, s.nombre as sucursal_nombre,
                       (SELECT MAX(r.fecha) FROM recetas r WHERE r.paciente_id = p.id AND r.empresa_id = p.empresa_id) as ultima_visita
                FROM pacientes p
                LEFT JOIN sucursales s ON p.sucursal_id = s.id
                WHERE $where
                ORDER BY p.apellido, p.nombre";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT p.*, s.nombre as sucursal_nombre
             FROM pacientes p
             LEFT JOIN sucursales s ON p.sucursal_id = s.id
             WHERE p.id = ? AND p.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function getRecetas(int $pacienteId): array
    {
        return $this->db->fetchAll(
            "SELECT r.*, CONCAT(u.nombre,' ',u.apellido) as profesional
             FROM recetas r
             LEFT JOIN usuarios u ON r.usuario_id = u.id
             WHERE r.paciente_id = ? AND r.empresa_id = ?
             ORDER BY r.fecha DESC",
            [$pacienteId, $this->empresaScope()]
        );
    }

    public function getVentas(int $pacienteId): array
    {
        return $this->db->fetchAll(
            "SELECT v.*, CONCAT(u.nombre,' ',u.apellido) as vendedor, s.nombre as sucursal_nombre
             FROM ventas v
             LEFT JOIN usuarios u ON v.usuario_id = u.id
             LEFT JOIN sucursales s ON v.sucursal_id = s.id
             WHERE v.paciente_id = ? AND v.empresa_id = ?
             ORDER BY v.fecha DESC",
            [$pacienteId, $this->empresaScope()]
        );
    }

    public function search(string $q): array
    {
        $s = "%$q%";
        return $this->db->fetchAll(
            "SELECT id, nombre, apellido, rut, telefono
             FROM pacientes
             WHERE empresa_id = ? AND estado = 'activo'
             AND (CONCAT(nombre,' ',apellido) LIKE ? OR rut LIKE ? OR telefono LIKE ?)
             ORDER BY apellido, nombre LIMIT 15",
            [$this->empresaScope(), $s, $s, $s]
        );
    }

    // ── Alias usados por el controlador ───────────────────────

    /**
     * Buscar pacientes con paginación.
     * Retorna ['data' => [...], 'pagination' => [...]]
     */
    public function buscar(string $search = '', int $page = 1, int $perPage = ITEMS_PER_PAGE): array
    {
        $result = $this->getAll($page, $search);
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    /**
     * Búsqueda simple para AJAX (alias de search)
     */
    public function buscarSimple(string $q): array
    {
        return $this->search($q);
    }

    /**
     * Crear nuevo paciente desde datos POST
     */
    public function crear(array $data): int
    {
        $campos = [
            'empresa_id'       => $this->empresaScope(),
            'sucursal_id'      => Auth::sucursalId() ?: null,
            'nombre'           => trim($data['nombre'] ?? ''),
            'apellido'         => trim($data['apellido'] ?? ''),
            'rut'              => trim($data['rut'] ?? '') ?: null,
            'fecha_nacimiento' => !empty($data['fecha_nacimiento']) ? $data['fecha_nacimiento'] : null,
            'genero'           => $data['genero'] ?? null,
            'telefono'         => trim($data['telefono'] ?? '') ?: null,
            'email'            => trim($data['email'] ?? '') ?: null,
            'direccion'        => trim($data['direccion'] ?? '') ?: null,
            'ciudad'           => trim($data['ciudad'] ?? '') ?: null,
            'observaciones'    => trim($data['observaciones'] ?? '') ?: null,
            'estado'           => 'activo',
        ];
        return $this->db->insert($this->table, $campos);
    }

    /**
     * Actualizar paciente desde datos POST
     */
    public function actualizar(int $id, array $data): int
    {
        $campos = [
            'nombre'           => trim($data['nombre'] ?? ''),
            'apellido'         => trim($data['apellido'] ?? ''),
            'rut'              => trim($data['rut'] ?? '') ?: null,
            'fecha_nacimiento' => !empty($data['fecha_nacimiento']) ? $data['fecha_nacimiento'] : null,
            'genero'           => $data['genero'] ?? null,
            'telefono'         => trim($data['telefono'] ?? '') ?: null,
            'email'            => trim($data['email'] ?? '') ?: null,
            'direccion'        => trim($data['direccion'] ?? '') ?: null,
            'ciudad'           => trim($data['ciudad'] ?? '') ?: null,
            'observaciones'    => trim($data['observaciones'] ?? '') ?: null,
        ];
        return $this->update($id, $campos);
    }
}
