<?php
/**
 * OptiCore SaaS - Stock Model
 */

class Stock extends BaseModel
{
    protected string $table = 'stock';

    // ── Alias con estructura ['data','pagination'] ─────────────
    public function listarStock(?int $sucursalId = null, ?string $busqueda = null, int $page = 1, int $perPage = ITEMS_PER_PAGE): array
    {
        $result = $this->getAll($page, $sucursalId, $busqueda ?? '');
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    public function listarMovimientos(?int $sucursalId = null, ?int $productoId = null, int $page = 1, int $perPage = ITEMS_PER_PAGE): array
    {
        $params = [$this->empresaScope()];
        $where  = 'ms.empresa_id = ?';

        if ($sucursalId) {
            $where   .= ' AND ms.sucursal_id = ?';
            $params[] = $sucursalId;
        }
        if ($productoId) {
            $where   .= ' AND ms.producto_id = ?';
            $params[] = $productoId;
        }

        $sql = "SELECT ms.*, p.nombre as producto_nombre, p.sku,
                       CONCAT(u.nombre,' ',u.apellido) as usuario_nombre,
                       s.nombre as sucursal_nombre
                FROM movimientos_stock ms
                JOIN productos p ON ms.producto_id = p.id
                LEFT JOIN usuarios u ON ms.usuario_id = u.id
                LEFT JOIN sucursales s ON ms.sucursal_id = s.id
                WHERE $where
                ORDER BY ms.fecha DESC, ms.id DESC";

        $result = $this->db->paginate($sql, $params, $page);
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    public function getAll(int $page = 1, ?int $sucursalId = null, string $search = ''): array
    {
        $params = [$this->empresaScope()];
        $where  = 'st.empresa_id = ?';

        if ($sucursalId) {
            $where   .= ' AND st.sucursal_id = ?';
            $params[] = $sucursalId;
        }
        if ($search) {
            $where   .= ' AND (p.nombre LIKE ? OR p.sku LIKE ?)';
            $s        = "%$search%";
            $params[] = $s; $params[] = $s;
        }

        $sql = "SELECT st.*, p.nombre as producto_nombre, p.sku, p.tipo,
                       p.stock_minimo, p.precio_venta, p.precio_costo,
                       c.nombre as categoria_nombre, m.nombre as marca_nombre,
                       s.nombre as sucursal_nombre
                FROM stock st
                JOIN productos p ON st.producto_id = p.id
                LEFT JOIN categorias c ON p.categoria_id = c.id
                LEFT JOIN marcas m ON p.marca_id = m.id
                JOIN sucursales s ON st.sucursal_id = s.id
                WHERE $where AND p.estado = 'activo'
                ORDER BY p.nombre ASC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getByProductoSucursal(int $productoId, int $sucursalId): array|false
    {
        return $this->db->fetchOne(
            "SELECT * FROM stock WHERE producto_id = ? AND sucursal_id = ? AND empresa_id = ?",
            [$productoId, $sucursalId, $this->empresaScope()]
        );
    }

    public function getCantidad(int $productoId, int $sucursalId): int
    {
        return (int) $this->db->fetchColumn(
            "SELECT COALESCE(cantidad, 0) FROM stock WHERE producto_id = ? AND sucursal_id = ? AND empresa_id = ?",
            [$productoId, $sucursalId, $this->empresaScope()]
        );
    }

    public function ajustar(int $productoId, int $sucursalId, int $cantidad, string $tipo, string $obs = '', ?int $usuarioId = null): bool
    {
        $db = $this->db;
        try {
            $db->beginTransaction();

            $stockActual = $this->getCantidad($productoId, $sucursalId);
            $nuevaCantidad = match($tipo) {
                'entrada', 'traspaso_entrada'  => $stockActual + $cantidad,
                'salida', 'venta', 'traspaso_salida' => $stockActual - $cantidad,
                'ajuste'   => $cantidad,
                default    => $stockActual + $cantidad,
            };

            if ($nuevaCantidad < 0) {
                $db->rollBack();
                return false;
            }

            // Upsert stock
            $exists = $db->exists('stock', ['producto_id' => $productoId, 'sucursal_id' => $sucursalId]);
            if ($exists) {
                $db->update('stock', ['cantidad' => $nuevaCantidad], [
                    'producto_id' => $productoId,
                    'sucursal_id' => $sucursalId,
                ]);
            } else {
                $db->insert('stock', [
                    'empresa_id'  => $this->empresaScope(),
                    'sucursal_id' => $sucursalId,
                    'producto_id' => $productoId,
                    'cantidad'    => $nuevaCantidad,
                ]);
            }

            // Registrar movimiento
            $db->insert('movimientos_stock', [
                'empresa_id'        => $this->empresaScope(),
                'sucursal_id'       => $sucursalId,
                'producto_id'       => $productoId,
                'usuario_id'        => Auth::id(),
                'tipo'              => $tipo,
                'cantidad'          => $cantidad,
                'cantidad_anterior' => $stockActual,
                'cantidad_nueva'    => $nuevaCantidad,
                'observaciones'     => $obs,
                'fecha'             => date('Y-m-d H:i:s'),
            ]);

            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollBack();
            error_log('Stock::ajustar error: ' . $e->getMessage());
            return false;
        }
    }

    public function transferir(int $productoId, int $sucursalOrigen, int $sucursalDestino, int $cantidad, string $obs = ''): bool
    {
        $db = $this->db;
        try {
            $db->beginTransaction();

            $stockOrigen = $this->getCantidad($productoId, $sucursalOrigen);
            if ($stockOrigen < $cantidad) {
                $db->rollBack();
                return false;
            }

            $stockDestino = $this->getCantidad($productoId, $sucursalDestino);
            $empresaId    = $this->empresaScope();

            // Reducir origen
            $db->update('stock', ['cantidad' => $stockOrigen - $cantidad], [
                'producto_id' => $productoId, 'sucursal_id' => $sucursalOrigen,
            ]);
            $db->insert('movimientos_stock', [
                'empresa_id' => $empresaId, 'sucursal_id' => $sucursalOrigen,
                'producto_id' => $productoId, 'usuario_id' => Auth::id(),
                'tipo' => 'traspaso_salida', 'cantidad' => $cantidad,
                'cantidad_anterior' => $stockOrigen, 'cantidad_nueva' => $stockOrigen - $cantidad,
                'observaciones' => "Traspaso a sucursal $sucursalDestino. $obs",
                'fecha' => date('Y-m-d H:i:s'),
            ]);

            // Aumentar destino
            $existsDest = $db->exists('stock', ['producto_id' => $productoId, 'sucursal_id' => $sucursalDestino]);
            if ($existsDest) {
                $db->update('stock', ['cantidad' => $stockDestino + $cantidad], [
                    'producto_id' => $productoId, 'sucursal_id' => $sucursalDestino,
                ]);
            } else {
                $db->insert('stock', [
                    'empresa_id' => $empresaId, 'sucursal_id' => $sucursalDestino,
                    'producto_id' => $productoId, 'cantidad' => $cantidad,
                ]);
            }
            $db->insert('movimientos_stock', [
                'empresa_id' => $empresaId, 'sucursal_id' => $sucursalDestino,
                'producto_id' => $productoId, 'usuario_id' => Auth::id(),
                'tipo' => 'traspaso_entrada', 'cantidad' => $cantidad,
                'cantidad_anterior' => $stockDestino, 'cantidad_nueva' => $stockDestino + $cantidad,
                'observaciones' => "Traspaso desde sucursal $sucursalOrigen. $obs",
                'fecha' => date('Y-m-d H:i:s'),
            ]);

            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollBack();
            return false;
        }
    }

    public function getKardex(int $productoId, ?int $sucursalId = null): array
    {
        $params = [$this->empresaScope(), $productoId];
        $where  = 'ms.empresa_id = ? AND ms.producto_id = ?';
        if ($sucursalId) {
            $where   .= ' AND ms.sucursal_id = ?';
            $params[] = $sucursalId;
        }
        return $this->db->fetchAll(
            "SELECT ms.*, CONCAT(u.nombre,' ',u.apellido) as usuario_nombre,
                    s.nombre as sucursal_nombre
             FROM movimientos_stock ms
             LEFT JOIN usuarios u ON ms.usuario_id = u.id
             LEFT JOIN sucursales s ON ms.sucursal_id = s.id
             WHERE $where
             ORDER BY ms.fecha DESC LIMIT 200",
            $params
        );
    }

    public function descontarVenta(int $ventaId, int $sucursalId): void
    {
        $items = $this->db->fetchAll(
            "SELECT producto_id, cantidad FROM venta_items WHERE venta_id = ?",
            [$ventaId]
        );
        foreach ($items as $item) {
            $this->ajustar($item['producto_id'], $sucursalId, $item['cantidad'], 'venta', "Venta #$ventaId");
        }
    }
}
