/**
 * OptiCore SaaS - JavaScript Principal
 * Bootstrap 5.3 + utilidades globales
 */

'use strict';

/* ── Sidebar toggle (mobile) ─────────────────────────────────── */
(function () {
    const sidebar  = document.querySelector('.oc-sidebar');
    const overlay  = document.querySelector('.oc-sidebar-overlay');
    const toggleBtn = document.querySelector('.btn-sidebar-toggle');

    if (!sidebar) return;

    function openSidebar() {
        sidebar.classList.add('show');
        overlay && overlay.classList.add('show');
        document.body.style.overflow = 'hidden';
    }

    function closeSidebar() {
        sidebar.classList.remove('show');
        overlay && overlay.classList.remove('show');
        document.body.style.overflow = '';
    }

    toggleBtn && toggleBtn.addEventListener('click', () => {
        sidebar.classList.contains('show') ? closeSidebar() : openSidebar();
    });

    overlay && overlay.addEventListener('click', closeSidebar);

    // Cerrar con Escape
    document.addEventListener('keydown', e => {
        if (e.key === 'Escape') closeSidebar();
    });
})();

/* ── Flash messages auto-dismiss ─────────────────────────────── */
(function () {
    const alerts = document.querySelectorAll('.flash-container .alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = bootstrap.Alert.getOrCreateInstance(alert);
            bsAlert && bsAlert.close();
        }, 5000);
    });
})();

/* ── Confirmación de eliminación ─────────────────────────────── */
document.addEventListener('click', function (e) {
    const btn = e.target.closest('[data-confirm]');
    if (!btn) return;
    const msg = btn.dataset.confirm || '¿Estás seguro?';
    if (!confirm(msg)) e.preventDefault();
});

/* ── AJAX para acciones destructivas con recarga automática ───── */
(function () {
    document.addEventListener('submit', function (e) {
        const form = e.target;
        if (!form.hasAttribute('data-confirm')) return;

        e.preventDefault();

        const msg = form.dataset.confirm || '¿Estás seguro?';
        if (!confirm(msg)) return;

        const formData = new FormData(form);

        fetch(form.action, {
            method: form.method || 'POST',
            body: formData,
        })
        .then(() => {
            window.location.reload();
        })
        .catch(() => {
            window.location.reload();
        });
    });
})();

/* ── Tooltips Bootstrap ──────────────────────────────────────── */
(function () {
    const tooltipEls = document.querySelectorAll('[data-bs-toggle="tooltip"]');
    tooltipEls.forEach(el => new bootstrap.Tooltip(el, { trigger: 'hover' }));
})();

/* ── Popovers Bootstrap ──────────────────────────────────────── */
(function () {
    const popoverEls = document.querySelectorAll('[data-bs-toggle="popover"]');
    popoverEls.forEach(el => new bootstrap.Popover(el));
})();

/* ── Formateo de moneda ──────────────────────────────────────── */
window.OptiCore = window.OptiCore || {};

OptiCore.formatMoney = function (amount, currency = 'CLP') {
    return new Intl.NumberFormat('es-CL', {
        style: 'currency',
        currency: currency,
        minimumFractionDigits: currency === 'CLP' ? 0 : 2,
    }).format(amount);
};

OptiCore.formatDate = function (dateStr) {
    if (!dateStr) return '-';
    const d = new Date(dateStr + 'T00:00:00');
    return d.toLocaleDateString('es-CL');
};

/* ── Búsqueda AJAX genérica ──────────────────────────────────── */
OptiCore.autocomplete = function ({
    inputEl,
    hiddenEl,
    suggestionsEl,
    url,
    minChars = 2,
    delay = 300,
    onSelect,
    renderItem,
}) {
    let timer;

    inputEl.addEventListener('input', function () {
        clearTimeout(timer);
        const q = this.value.trim();
        if (q.length < minChars) {
            suggestionsEl.style.display = 'none';
            return;
        }
        timer = setTimeout(() => {
            fetch(url + encodeURIComponent(q))
                .then(r => r.json())
                .then(data => {
                    suggestionsEl.innerHTML = '';
                    if (!data.length) { suggestionsEl.style.display = 'none'; return; }
                    data.forEach(item => {
                        const a = document.createElement('a');
                        a.href = '#';
                        a.className = 'list-group-item list-group-item-action small';
                        a.innerHTML = renderItem ? renderItem(item) : item.nombre;
                        a.addEventListener('click', e => {
                            e.preventDefault();
                            onSelect(item);
                            suggestionsEl.style.display = 'none';
                        });
                        suggestionsEl.appendChild(a);
                    });
                    suggestionsEl.style.display = 'block';
                })
                .catch(() => { suggestionsEl.style.display = 'none'; });
        }, delay);
    });

    document.addEventListener('click', e => {
        if (!suggestionsEl.contains(e.target) && e.target !== inputEl) {
            suggestionsEl.style.display = 'none';
        }
    });

    inputEl.addEventListener('keydown', e => {
        if (e.key === 'Escape') suggestionsEl.style.display = 'none';
    });
};

/* ── Preview de imagen al seleccionar archivo ────────────────── */
OptiCore.imagePreview = function (inputId, previewId) {
    const input   = document.getElementById(inputId);
    const preview = document.getElementById(previewId);
    if (!input || !preview) return;

    input.addEventListener('change', function () {
        if (this.files && this.files[0]) {
            const reader = new FileReader();
            reader.onload = e => {
                preview.src = e.target.result;
                preview.classList.remove('d-none');
            };
            reader.readAsDataURL(this.files[0]);
        }
    });
};

/* ── Toggle visibilidad de contraseña ────────────────────────── */
OptiCore.passwordToggle = function (inputId, btnId) {
    const input = document.getElementById(inputId);
    const btn   = document.getElementById(btnId);
    if (!input || !btn) return;

    btn.addEventListener('click', () => {
        const isText = input.type === 'text';
        input.type = isText ? 'password' : 'text';
        btn.querySelector('i').className = isText ? 'bi bi-eye' : 'bi bi-eye-slash';
    });
};

/* ── Contador de caracteres ──────────────────────────────────── */
OptiCore.charCounter = function (inputId, counterId, max) {
    const input   = document.getElementById(inputId);
    const counter = document.getElementById(counterId);
    if (!input || !counter) return;

    function update() {
        const len = input.value.length;
        counter.textContent = `${len}/${max}`;
        counter.className = len > max * .9 ? 'form-text text-warning' : 'form-text text-muted';
    }
    input.addEventListener('input', update);
    update();
};

/* ── Imprimir página ─────────────────────────────────────────── */
OptiCore.print = function () {
    window.print();
};

/* ── Confirmar antes de salir con cambios ────────────────────── */
OptiCore.confirmLeave = function (formId) {
    const form = document.getElementById(formId);
    if (!form) return;

    let changed = false;
    form.addEventListener('change', () => { changed = true; });
    form.addEventListener('submit', () => { changed = false; });

    window.addEventListener('beforeunload', e => {
        if (changed) {
            e.preventDefault();
            e.returnValue = '';
        }
    });
};

/* ── Seleccionar todo el texto al hacer focus ────────────────── */
document.querySelectorAll('input[data-select-all]').forEach(input => {
    input.addEventListener('focus', function () { this.select(); });
});

/* ── Inicializar componentes al cargar ───────────────────────── */
document.addEventListener('DOMContentLoaded', function () {
    // Activar todos los tooltips
    document.querySelectorAll('[data-bs-toggle="tooltip"]').forEach(el => {
        new bootstrap.Tooltip(el);
    });

    // Auto-focus en primer input de formularios
    const firstInput = document.querySelector('form input:not([type=hidden]):not([readonly])');
    if (firstInput && !firstInput.value) {
        // Solo auto-focus si no hay valor (no re-enfocar en edición)
    }

    // Marcar nav link activo
    const currentPath = window.location.pathname;
    document.querySelectorAll('.oc-nav-link').forEach(link => {
        const href = link.getAttribute('href');
        if (href && href !== '/' && currentPath.startsWith(href)) {
            link.classList.add('active');
        }
    });
});

/* ── Utilidad: debounce ──────────────────────────────────────── */
OptiCore.debounce = function (fn, delay = 300) {
    let timer;
    return function (...args) {
        clearTimeout(timer);
        timer = setTimeout(() => fn.apply(this, args), delay);
    };
};

/* ── Utilidad: copiar al portapapeles ────────────────────────── */
OptiCore.copyToClipboard = function (text, btn) {
    navigator.clipboard.writeText(text).then(() => {
        if (btn) {
            const original = btn.innerHTML;
            btn.innerHTML = '<i class="bi bi-check-lg"></i>';
            setTimeout(() => { btn.innerHTML = original; }, 1500);
        }
    });
};
